//
// proj_paint_cone.fx
//

//------------------------------------------------------------------------------------------
// Settings
//------------------------------------------------------------------------------------------
float3 elementPosition = float3(0,0,0);
float3 elementFwVector = float3(0,0,0);
float3 elementUpVector = float3(0,0,0);

float2 sPicPos = float2(0,0);
float2 sPicSize = float2(1,1);
bool sPicBright = false;
int sTexAddress = 4;
bool sMask = true;
float sMaskHeight = 0;
bool sEnableDirt = false;
bool sEnableEnv = false;

texture gTextureColor;

//------------------------------------------------------------------------------------------
// Include some common stuff
//------------------------------------------------------------------------------------------
int gCapsMaxAnisotropy < string deviceCaps="MaxAnisotropy"; >;
#define GENERATE_NORMALS
#include "mta-helper.fx"

//------------------------------------------------------------------------------------------
// Sampler for the main texture
//------------------------------------------------------------------------------------------
sampler Sampler0 = sampler_state
{
    Texture = (gTexture0);
};

sampler Sampler1 = sampler_state
{
    Texture = (gTexture1);
};

sampler SamplerColor = sampler_state
{
    Texture = (gTextureColor);
    MipFilter = Linear;
    MaxAnisotropy = gCapsMaxAnisotropy;
    MinFilter = Anisotropic;
    AddressU = sTexAddress;
    AddressV = sTexAddress;
    BorderColor = float4(0,0,0,0);
};

//------------------------------------------------------------------------------------------
// Structure of data sent to the vertex shader
//------------------------------------------------------------------------------------------
struct VSInput
{
  float3 Position : POSITION0;
  float3 Normal : NORMAL0;
  float4 Diffuse : COLOR0;
  float2 TexCoord : TEXCOORD0;
  float2 TexCoord1 : TEXCOORD1;
};

//------------------------------------------------------------------------------------------
// Structure of data sent to the pixel shader ( from the vertex shader )
//------------------------------------------------------------------------------------------
struct PSInput
{
  float4 Position : POSITION0;
  float4 Diffuse : COLOR0;
  float3 Specular : COLOR1;
  float2 TexCoord : TEXCOORD0;
  float2 TexCoord1 : TEXCOORD1;
  float2 TexCoord2 : TEXCOORD2;
  float2 TexCoord3 : TEXCOORD3;
  float PointPlaneDist : TEXCOORD4;
};

//------------------------------------------------------------------------------------------
// Create view matrix 
//------------------------------------------------------------------------------------------
float4x4 createViewMatrix( float3 pos, float3 fwVec, float3 upVec )
{
    float3 zaxis = normalize( fwVec );    // The "forward" vector.
    float3 xaxis = normalize( cross( -upVec, zaxis ));// The "right" vector.
    float3 yaxis = cross( xaxis, zaxis );     // The "up" vector.

    // Create a 4x4 view matrix from the right, up, forward and eye position vectors
    float4x4 viewMatrix = {
        float4(      xaxis.x,            yaxis.x,            zaxis.x,       0 ),
        float4(      xaxis.y,            yaxis.y,            zaxis.y,       0 ),
        float4(      xaxis.z,            yaxis.z,            zaxis.z,       0 ),
        float4(-dot( xaxis, pos ), -dot( yaxis, pos ), -dot( zaxis, pos ),  1 )
    };
    return viewMatrix;
}

//------------------------------------------------------------------------------------------
// Create orthographic projection matrix 
//------------------------------------------------------------------------------------------
float4x4 createPerspectiveProjectionMatrix(float near_plane, float far_plane, float fov_horiz, float fov_aspect)
{
    float h, w, Q;

    w = 1/tan(fov_horiz * 0.5);
    h = w / fov_aspect;
    Q = far_plane/(far_plane - near_plane);

    float4x4 projectionMatrix = {
        float4(      w,            0,        0,             0 ),
        float4(      0,            h,        0,             0 ),
        float4(      0,            0,        Q,             1 ),
        float4(      0,            0,        -Q*near_plane, 0 )
    };    
    return projectionMatrix;
}

//------------------------------------------------------------------------------------------
// lightEnableState

int gLight0Enable           < string lightEnableState="0,Enable"; >;
int gLight1Enable           < string lightEnableState="1,Enable"; >;
int gLight2Enable           < string lightEnableState="2,Enable"; >;
int gLight3Enable           < string lightEnableState="3,Enable"; >;
int gLight4Enable           < string lightEnableState="4,Enable"; >;

// lightState 
float4 gLight0Diffuse           < string lightState="0,Diffuse"; >;
float3 gLight0Direction         < string lightState="0,Direction"; >;
float4 gLight1Diffuse           < string lightState="1,Diffuse"; >;
float3 gLight1Direction         < string lightState="1,Direction"; >;
float4 gLight2Diffuse           < string lightState="2,Diffuse"; >;
float3 gLight2Direction         < string lightState="2,Direction"; >;
float4 gLight3Diffuse           < string lightState="3,Diffuse"; >;
float3 gLight3Direction         < string lightState="3,Direction"; >;
float4 gLight4Diffuse           < string lightState="4,Diffuse"; >;
float3 gLight4Direction         < string lightState="4,Direction"; >;

//------------------------------------------------------------------------------------------
// MTACalcGTACompleteDiffuse
// - Calculate GTA lighting including pointlights for vehicles (all 4 lights)
//------------------------------------------------------------------------------------------
float4 MTACalcGTACompleteDiffuse( float3 WorldNormal, float4 InDiffuse )
{
    // Calculate diffuse color by doing what D3D usually does
    float4 ambient  = gAmbientMaterialSource  == 0 ? gMaterialAmbient  : InDiffuse;
    float4 diffuse  = gDiffuseMaterialSource  == 0 ? gMaterialDiffuse  : InDiffuse;
    float4 emissive = gEmissiveMaterialSource == 0 ? gMaterialEmissive : InDiffuse;

    float4 TotalAmbient = ambient * ( gGlobalAmbient + gLightAmbient );

    // Add all the 4 pointlights
    float DirectionFactor=0;
    float4 TotalDiffuse=0;
	
    if (gLight1Enable) {
    DirectionFactor = max(0,dot(WorldNormal, -gLight1Direction ));
    TotalDiffuse += ( gLight1Diffuse * DirectionFactor );
                     }
    if (gLight2Enable) {
    DirectionFactor = max(0,dot(WorldNormal, -gLight2Direction ));
    TotalDiffuse += ( gLight2Diffuse * DirectionFactor );
                     }
    if (gLight3Enable) {
    DirectionFactor = max(0,dot(WorldNormal, -gLight3Direction ));
    TotalDiffuse += ( gLight3Diffuse * DirectionFactor );
                     }
    if (gLight4Enable) {
    DirectionFactor = max(0,dot(WorldNormal, -gLight4Direction ));
    TotalDiffuse += ( gLight4Diffuse * DirectionFactor );
                     }	
					 
    TotalDiffuse *= diffuse;
	
    float4 OutDiffuse = saturate(TotalDiffuse + TotalAmbient + emissive);
    OutDiffuse.a *= diffuse.a;

    return OutDiffuse;
}

//------------------------------------------------------------------------------------------
// Get Projected TexCoords 
//------------------------------------------------------------------------------------------
float2 getProjectedTexCoord(float4 projPos, float2 uv)
{
    projPos.xy += uv.xy;
    float projX = (0.5 * (projPos.w + projPos.x));
    float projY = (0.5 * (projPos.w - projPos.y));
    return float2(projX, projY) / projPos.w;
}

//------------------------------------------------------------------------------------------
// VertexShaderFunction
//------------------------------------------------------------------------------------------
PSInput VertexShaderFunction(VSInput VS)
{
    PSInput PS = (PSInput)0;

    // Calculate screen pos of vertex and pass to pixel shader
    PS.Position = mul(float4(VS.Position, 1), gWorldViewProjection);

    // Create view and projection matrices (for the projective texture)	
    float camDist = min(max(sPicSize.x, sPicSize.y), 1);
    float4x4 sView = createViewMatrix(elementPosition + gWorld[2].xyz * 1.5 / camDist, - gWorld[2].xyz, gWorld[1].xyz);
    float4x4 sViewMask = createViewMatrix(elementPosition + elementUpVector * 1.5 / camDist, -elementUpVector , elementFwVector);
    float4x4 sProjection = createPerspectiveProjectionMatrix( -0.3, 800, 1.2217, sPicSize.y / sPicSize.x);

    // Calculate the projective texture coordinates
    float4 worldPos = mul(float4(VS.Position.xyz,1), gWorld);
    float4 viewPos = mul(worldPos, sView);
    float4 projPos = mul(viewPos, sProjection);
	
    float4 viewPosMask = mul(worldPos, sViewMask);
    float4 projPosMask = mul(viewPosMask, sProjection); 

    // Pass texture coordinates to pixel shader	
    PS.TexCoord = VS.TexCoord;
    PS.TexCoord1 = VS.TexCoord1;
	
    // Pass projective texture coordinates to pixel shader
    PS.TexCoord2 = getProjectedTexCoord(projPos, sPicPos);
    PS.TexCoord3 = getProjectedTexCoord(projPosMask, sPicPos);

    // Calculate GTA lighting for vehicle
    float3 worldNormal = mul(VS.Normal, (float3x3)gWorld).xyz;
    PS.Diffuse = MTACalcGTACompleteDiffuse(worldNormal, VS.Diffuse);
	PS.Specular.rgb = saturate(gMaterialSpecular.rgb * MTACalculateSpecular(gCameraDirection, gLight1Direction, worldNormal, gMaterialSpecPower) * 0.75);	

    // Texture mask
    float3 upVec = normalize(elementUpVector);
    PS.PointPlaneDist = dot(upVec.xyz, worldPos.xyz - elementPosition);

    return PS;
}

//------------------------------------------------------------------------------------------
// PixelShaderFunction
//------------------------------------------------------------------------------------------
float4 PixelShaderFunction(PSInput PS) : COLOR0
{
    // Get Paintjob texture
    float4 texel = tex2D(SamplerColor, PS.TexCoord2.xy);

    // Mask the texture
    if ((PS.PointPlaneDist < sMaskHeight) && (sMask)) texel.a = 0;

    if (sTexAddress == 4)
	    if ((PS.TexCoord3.x < 0) || (PS.TexCoord3.x > 1) || (PS.TexCoord3.y < 0) || (PS.TexCoord3.y > 1)) texel.a = 0; 

    // Multiply by paint specular color	
    if (!sPicBright) texel.rgb *= saturate(0.1 + gMaterialSpecular.rgb * 1.25);

    // Get vehicle paint color	
    float4 finalColor = PS.Diffuse;

    // Get and apply env map texture
    float3 refTex = tex2D(Sampler1, PS.TexCoord1.xy).rgb;
    if (sEnableEnv) finalColor.rgb += saturate(refTex.rgb * gMaterialSpecular.rgb * 0.22);

    // Lerp betwen paint and texture
    finalColor.rgb = lerp(finalColor.rgb, texel.rgb, texel.a);

    // Apply specular highlight
    finalColor.rgb += PS.Specular.rgb;
	
    // Get and apply dirt texture
    float3 dirtTex = tex2D(Sampler0, PS.TexCoord.xy).rgb;
    if (sEnableDirt) finalColor.rgb *= dirtTex;

    // Pass the outcome
    finalColor.rgb = saturate(finalColor.rgb);
    finalColor.a = PS.Diffuse.a;

    return finalColor;
}

//------------------------------------------------------------------------------------------
// Techniques
//------------------------------------------------------------------------------------------
technique perspective_projection
{
    pass P0
    {
        VertexShader = compile vs_2_0 VertexShaderFunction();
        PixelShader = compile ps_2_0 PixelShaderFunction();
    }
}

// Fallback
technique fallback
{
    pass P0
    {
        // Just draw normally
    }
}
